/**
 * @file
 * @date 11.11.2013
 * @brief AnaPico device communication library.
 * Adress strings:
 * Socket LAN:		[TCP-]###.###.###.### (IP address) or [TCP-]###-#########-#### (serial number)
 * USBTMC:			USB-###-#########-#### (serial number)
 * VISA:			VISA resource name string
 */

#ifndef _AP_COMM_LIB_H_
#define _AP_COMM_LIB_H_

// DLL export macro
#ifdef _cplusplus
#ifndef DLL_AP_COMM
#define DLL_AP_COMM extern "C" __declspec(dllexport)
#endif
#endif
#ifdef __cplusplus
#ifndef DLL_AP_COMM
#define DLL_AP_COMM extern "C" __declspec(dllexport)
#endif
#endif
#ifndef DLL_AP_COMM
#define DLL_AP_COMM extern __declspec(dllexport)
#endif

// Success/error codes
#define VI_SUCCESS				0x00000000		// Operation completed successfully
#define VI_SUCCESS_MAX_CNT		0x3FFF0006		// The number of bytes read is equal to the input count, more data available
#define VI_ERROR_SYSTEM_ERROR	0xBFFF0000		// Unknown system error
#define VI_ERROR_TMO			0xBFFF0015		// Timeout expired before operation completed
#define VI_ERROR_CONN_LOST		0xBFFF00A6		// The connection for the given session has been lost
#define VI_ERROR_FILE_ACCESS	0xBFFF00A1		// Accessing the file system failed

// Interface selection macros
#define AP_COMM_IFACE_LAN		0x00000001		// Interface selection LAN (15 unused codes for future LAN protocols, e.g. socket or VXI-11)
#define AP_COMM_IFACE_USB		0x00000010		// Interface selection USB (15 unused codes for future USB protocols)
#define AP_COMM_IFACE_VISA_GPIB	0x00000200		// Interface selection VISA (GPIB only)

/**
 * Loads the external VISA libraries and enables VISA support. VISA support is required
 * to communicate with GPIB instruments.
 * @return						VI_SUCCESS in case of success, VI_ERROR_SYSTEM_ERROR if
 *								no installed VISA library has been found.
 */
DLL_AP_COMM int ap_comm_loadVisa(void);

/**
 * Frees the external VISA libraries and disables VISA support.
 */
DLL_AP_COMM void ap_comm_freeVisa(void);

/**
 * Finds AnaPico devices. Follows the cdecl calling convention.
 * @param[in,out]	devices		Buffer holding the found devices string.
 *								String format:
 *								Model1:SerialNo1:Address1;Model2:SerialNo2:Address2;...
 * @param[in]		tmoMs		Find timeout in milliseconds, 500 is a reasonable value.
 *								Decrease it to speed up the find process, increase it if
 *								devices aren't found sometimes.
 * @param[in]		count		Size of the devices buffer, in bytes.
 * @param[out]		retCount	Returns the number of bytes written to the devices buffer.
 * @param[in]		iface		Interface selection, bitwise OR of one or more of the following macros.
 *								AP_COMM_IFACE_LAN: Find devices connected via LAN.
 *								AP_COMM_IFACE_USB: Find devices connected via USB.
 * @return						VI_SUCCESS in case of success, VI_ERROR code otherwise.
 */
DLL_AP_COMM int ap_comm_find(char* devices, int tmoMs, int count, int* retCount, int iface);

/**
 * Opens a connection to an AnaPico USB device. It's possible to pass the address returned by
 * ap_comm_find() or create an address string manually.
 * LAN address format: "[TCP-]<IPv4 address>, e.g. "TCP-192.168.1.2" or "192.168.1.2".
 * USB address format: "USB-<SN>" where <SN> is the serial number, e.g. "USB-121-213300010-0093".
 * ap_comm_open() follows the cdecl calling convention.
 * stdcall_ap_comm_open() follows the stdcall calling convention.
 * @param[in]		address		Device address.
 * @param[in]		tmoMs		Initial connection timeout in milliseconds, 2000 is a reasonable value.
 * @param[out]		retLinkId	Link ID >=1 in case of success, -1 otherwise.
 * @return						VI_SUCCESS in case of success, VI_ERROR code otherwise.
 */
DLL_AP_COMM int ap_comm_open(char* address, int tmoMs, int* retLinkId);

/**
 * Closes a connection to an AnaPico USB device.
 * ap_comm_close() follows the cdecl calling convention.
 * stdcall_ap_comm_close() follows the stdcall calling convention.
 * @param[in]		linkId		Link ID  as returned by ap_comm_open().
 * @return						VI_SUCCESS in case of success, VI_ERROR code otherwise.
 */
DLL_AP_COMM int ap_comm_close(int linkId);

/**
 * Sends a command or query to an AnaPico device.
 * ap_comm_puts() follows the cdecl calling convention.
 * stdcall_ap_comm_puts() follows the stdcall calling convention.
 * @param[in]		linkId		Link ID returned by ap_comm_open().
 * @param[in]		cmd			Command string. Can be a terminated by '\0' or '\n'.
 * @param[in]		count		Size of the command string, in bytes. Zero can be passed
 *								if the command string is terminated by '\0' or '\n'.
 * @param[out]		retCount	Returns the number of bytes actually sent to the device.
 *								NULL can be passed.
 * @return						VI_SUCCESS in case of success, VI_ERROR code otherwise.
 */
DLL_AP_COMM int ap_comm_puts(int linkId, char* cmd, int count, int* retCount);

/**
 * Receives a query response from an AnaPico device.
 * ap_comm_gets() follows the cdecl calling convention.
 * stdcall_ap_comm_gets() follows the stdcall calling convention.
 * @param[in]		linkId		Link ID returned by ap_comm_open().
 * @param[out]		ans			Response buffer. Respones are terminated by '\n' and '\0'.
 * @param[in]		count		Size of the response buffer, in bytes.
 * @param[out]		retCount	Returns the number of bytes actually read from to the device.
 *								NULL can be passed.
 * @return						VI_SUCCESS in case of success, VI_ERROR code otherwise.
 */
DLL_AP_COMM int ap_comm_gets(int linkId, char* ans, int count, int* retCount);

/**
 * Sends binary data to an AnaPico USB device, e.g. for sending multiple commands at once.
 * Unlike ap_usb_puts(), this function does not care about command termination.
 * @param[in]		linkId		Link ID returned by ap_comm_open().
 * @param[in]		buffer		Data.
 * @param[in]		count		Data size in bytes.
 * @param[out]		retCount	Returns the number of bytes actually sent to the device.
 *								NULL can be passed.
 * @return						VI_SUCCESS in case of success, VI_ERROR code otherwise.
 */
DLL_AP_COMM int ap_comm_write(int linkId, void* buffer, int count, int* retCount);
DLL_AP_COMM int ap_comm_writeBlocking(int linkId, void* buffer, int count, int* retCount);

/**
 * Reads binary response data from an AnaPico USB device.
 * Unlike ap_usb_gets(), this function does not care about command termination.
 * It is not guaranteed that this function returns a complete response. Thus
 * it's in the callers responsibility to repeat the read calls until all data
 * has been read.
 * @param[in]		linkId		Link ID returned by ap_comm_open().
 * @param[out]		buffer		Response buffer.
 * @param[in]		count		Size of the response buffer, in bytes.
 * @param[out]		retCount	Returns the number of bytes actually read from to the device.
 * @return						VI_SUCCESS in case of success, VI_ERROR code otherwise.
 */
DLL_AP_COMM int ap_comm_read(int linkId, void* buffer, int count, int* retCount);
DLL_AP_COMM int ap_comm_readBlocking(int linkId, void* buffer, int count, int* retCount);

/**
 * Sends a binary data block to an AnaPico device. The data is retrieved from a buffer.
 * Uses the IEEE488.2 definite length arbitrary block data format.
 * @param[in]		linkId		Link ID returned by ap_comm_open().
 * @param[in]		cmd			Command preceding the binary data block.
 * @param[in]		count		Size of the command string, in bytes. Zero can be passed
 *								if the command string is terminated by NULL '\0' or NL '\n'.
 * @param[out]		retCount	Returns the number of command bytes actually sent to the device.
 *								NULL can be passed.
 * @param[in]		buf			Data to be sent.
 * @param[in]		size		Number of data bytes to be sent.
 * @return						VI_SUCCESS in case of success, VI_ERROR code otherwise.
 */
DLL_AP_COMM int ap_comm_writeBlock(int linkId, char* cmd, void* buf, int size);

/**
 * Sends a binary data block to an AnaPico device. The data is retrieved from a file.
 * Uses the IEEE488.2 definite length arbitrary block data format.
 * @param[in]		linkId		Link ID returned by ap_comm_open().
 * @param[in]		cmd			Command string preceding the binary data block.
 * @param[in]		count		_s functions only: size of the command string, in bytes. Zero can be passed
 *								if the command string is terminated by NULL '\0' or NL '\n'.
 * @param[out]		retCount	Returns the number of command bytes actually sent to the device.
 *								NULL can be passed.
 * @param[in]		filename	String containing path and name of file containing data to be sent.
 * @param[out]		len			Size of the filename string, in bytes. Zero can be passed
								if the filename string is terminated by NULL '\0' or NL '\n'.
 * @return						VI_SUCCESS in case of success, VI_ERROR code otherwise.
 */
DLL_AP_COMM int ap_comm_writeBlockFile(int linkId, char* cmd, char* filename);

/**
 * Returns the size of a binary data block from an AnaPico device. Call ap_comm_readBlockData() to
 * get the block data.
 * Supports the IEEE488.2 definite length arbitrary block and AnaPicos proprietary block data format.
 * @param[in]		linkId		Link ID returned by ap_comm_open().
 * @param[in]		cmd			Command returning the binary data block.
 * @param[out]		retSize		Size of the binary data block in bytes.
 * @return						VI_SUCCESS in case of success, VI_ERROR code otherwise.
 */
DLL_AP_COMM int ap_comm_readBlockSize(int linkId, char* cmd, int* retSize);

/**
 * Reads block data from an AnaPico device after querying the block size by calling
 * ap_comm_readBlockSize(). The data will be written to a buffer.
 * Supports the IEEE488.2 definite length arbitrary block and AnaPicos proprietary block data format.
 * @param[in]		linkId		Link ID returned by ap_comm_open().
 * @param[out]		buffer		Buffer holding received data.
 * @param[in]		size		Size of the buffer in bytes.
 * @param[out]		retCount	Number of bytes received.
 * @return						VI_SUCCESS or VI_SUCCESS_MAX_CNT in case of success, VI_ERROR code
 *								otherwise. VI_SUCCESS_MAX_CNT indicates that there is more data
 *								available. Repeat calling this function until it returns
 *								VI_SUCCESS.
 */
DLL_AP_COMM int ap_comm_readBlockData(int linkId, void* buffer, int count, int* retCount);

/**
 * Reads a binary data block from an AnaPico device. The data will be written to a buffer.
 * Supports the IEEE488.2 definite length arbitrary block and AnaPicos proprietary block data format.
 * @param[in]		linkId		Link ID returned by ap_comm_open().
 * @param[in]		cmd			Command returning the binary data block.
 * @param[out]		buffer		Buffer holding received data.
 * @param[in]		size		Size of the buffer in bytes.
 * @param[out]		retSize		Number of bytes received.
 * @return						VI_SUCCESS or VI_SUCCESS_MAX_CNT in case of success, VI_ERROR code
 *								otherwise. VI_SUCCESS_MAX_CNT indicates that there is more data
 *								available. Repeat calling this function until it returns
 *								VI_SUCCESS.
 */
DLL_AP_COMM int ap_comm_readBlock(int linkId, char* cmd, void* buffer, int count, int* retCount);

/**
 * Reads a binary data block from an AnaPico device. The data will be written to a file.
 * Supports the IEEE488.2 definite length arbitrary block and AnaPicos proprietary block data format.
 * @param[in]		linkId		Link ID returned by ap_comm_open().
 * @param[in]		cmd			Command returning the binary data block.
 * @param[in]		filename	Path and name of file to be stored. An existing file will be overwritten.
 * @param[out]		retSize		Number of bytes received.
 * @return						VI_SUCCESS in case of success, VI_ERROR code otherwise.
 */
DLL_AP_COMM int ap_comm_readBlockFile(int linkId, char* cmd, char* filename);

/**
 * Clears all link buffers. Clears data pending to be sent or received. 
 * @param[in]		linkId		Link ID returned by ap_comm_open().
 * @return						VI_SUCCESS in case of success, VI_ERROR code otherwise.
 */
DLL_AP_COMM int ap_comm_clear(int linkId);

/**
 * Sets the connection timeout in milliseconds. 
 * @param[in]		linkId		Link ID returned by ap_comm_open().
 * @param[in]		tmoMs		Timeout in milliseconds. Default is 2000.
 * @return						VI_SUCCESS in case of success, VI_ERROR code otherwise.
 */
DLL_AP_COMM int ap_comm_setTmoMs(int linkId, int tmoMs);

#endif
